<?php
/**
 * Plugin Name: B19 AI Bot Chat
 * Plugin URI: https://b19.co.jp/ai-bot-chat
 * Description: AI-powered chatbot plugin with full Japanese support. Developed in Japan for Japanese websites. 日本語完全対応のAIチャットボットプラグイン。
 * Version: 1.5.0
 * Author: 株式会社ビーク
 * Author URI: https://b19.co.jp/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: b19-ai-bot-chat
 * Network: true
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 */

// 直接アクセスを防ぐ
if (!defined('ABSPATH')) {
    exit;
}

// プラグイン定数
define('B19COJP_AI_BOT_CHAT_VERSION', '1.5.0');
define('B19COJP_AI_BOT_CHAT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('B19COJP_AI_BOT_CHAT_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('B19COJP_AI_BOT_CHAT_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * AI Bot Chat メインクラス
 */
class B19COJP_AI_Bot_Chat_Main {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // 初期化
        add_action('init', array($this, 'init'));
        
        // 管理画面
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'admin_init'));
        
        // フロントエンド
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_footer', array($this, 'render_chat_widget'));
        
        // ショートコード
        add_shortcode('b19cojp_ai_bot_chat', array($this, 'b19cojp_ai_bot_chat_shortcode'));
        
        // Ajax
        add_action('wp_ajax_b19cojp_ai_bot_chat_send_message', array($this, 'ajax_send_message'));
        add_action('wp_ajax_nopriv_b19cojp_ai_bot_chat_send_message', array($this, 'ajax_send_message'));
        
        // プラグイン有効化・無効化
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    public function init() {
        // 必要なクラスを読み込み
        require_once B19COJP_AI_BOT_CHAT_PLUGIN_PATH . 'includes/class-openai-client.php';
        require_once B19COJP_AI_BOT_CHAT_PLUGIN_PATH . 'includes/class-rate-limit.php';
        require_once B19COJP_AI_BOT_CHAT_PLUGIN_PATH . 'includes/class-license-validator.php';
        require_once B19COJP_AI_BOT_CHAT_PLUGIN_PATH . 'includes/class-google-sheets-client.php';
        require_once B19COJP_AI_BOT_CHAT_PLUGIN_PATH . 'includes/class-google-oauth.php';
        
        // Google OAuth初期化
        new B19AIBOC_Google_OAuth();
    }
    
    public function add_admin_menu() {
        add_menu_page(
            __('B19 AI Bot Chat', 'b19-ai-bot-chat'),
            __('B19 AI Bot Chat', 'b19-ai-bot-chat'),
            'manage_options',
            'b19aiboc-settings',
            array($this, 'admin_page'),
            'dashicons-format-chat',
            30
        );
        
        add_submenu_page(
            'b19aiboc-settings',
            __('Settings', 'b19-ai-bot-chat'),
            __('Settings', 'b19-ai-bot-chat'),
            'manage_options',
            'b19aiboc-settings',
            array($this, 'admin_page')
        );
    }
    
    public function admin_init() {
        // 設定の登録
        register_setting('b19cojp_ai_bot_chat_options', 'b19cojp_ai_bot_chat_options', array($this, 'sanitize_options'));
    }
    
    public function admin_page() {
        // 権限チェック
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('このページにアクセスする権限がありません。', 'b19-ai-bot-chat'));
        }
        
        $b19cojp_ai_bot_chat_options = get_option('b19cojp_ai_bot_chat_options', array());
        include B19COJP_AI_BOT_CHAT_PLUGIN_PATH . 'admin/views/settings-page.php';
    }
    
    public function enqueue_scripts() {
        if (!is_admin()) {
            $options = get_option('b19cojp_ai_bot_chat_options', array());
            
            // プラグインが有効化されているかチェック
            if (!isset($options['enabled']) || !$options['enabled']) {
                return;
            }
            
            // CSS
            wp_enqueue_style(
                'b19cojp-ai-bot-chat-chatbot',
                B19COJP_AI_BOT_CHAT_PLUGIN_URL . 'assets/css/chatbot-fixes.css',
                array(),
                B19COJP_AI_BOT_CHAT_VERSION
            );
            
            // JavaScript
            wp_enqueue_script(
                'b19cojp-ai-bot-chat-chatbot',
                B19COJP_AI_BOT_CHAT_PLUGIN_URL . 'assets/js/frontend.js',
                array('jquery'),
                B19COJP_AI_BOT_CHAT_VERSION,
                true
            );
            
            // AJAX設定をローカライズ
            wp_localize_script('b19cojp-ai-bot-chat-chatbot', 'b19cojp_ai_bot_chat_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('b19cojp_ai_bot_chat_nonce')
            ));
            
        }
    }
    
    public function b19cojp_ai_bot_chat_shortcode($atts) {
        // 重複防止：既に表示済みの場合は空を返す
        static $already_rendered = false;
        if ($already_rendered) {
            return '';
        }
        $already_rendered = true;
        
        $atts = shortcode_atts(array(
            'type' => 'chat',
            'height' => '400px'
        ), $atts);
        
        // プラグインが有効化されているかチェック
        $options = get_option('b19cojp_ai_bot_chat_options', array());
        if (!isset($options['enabled']) || !$options['enabled']) {
            return '';
        }
        
        ob_start();
        include B19COJP_AI_BOT_CHAT_PLUGIN_PATH . 'templates/chat-interface.php';
        return ob_get_clean();
    }
    
    public function render_chat_widget() {
        $options = get_option('b19cojp_ai_bot_chat_options', array());
        
        // プラグインが有効化されているかチェック
        if (!isset($options['enabled']) || !$options['enabled']) {
            return;
        }
        
        // ショートコードで表示
        echo do_shortcode('[b19cojp_ai_bot_chat]');
    }
    
    public function ajax_send_message() {
        // セキュリティチェック
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'b19cojp_ai_bot_chat_nonce')) {
            wp_send_json_error(__('セキュリティエラー', 'b19-ai-bot-chat'));
        }
        
        if (!isset($_POST['message'])) {
            wp_send_json_error(__('メッセージが指定されていません', 'b19-ai-bot-chat'));
        }
        
        $message = sanitize_textarea_field(wp_unslash($_POST['message']));
        
        // メッセージの長さをチェック
        if (strlen($message) > 1000) {
            wp_send_json_error(__('メッセージが長すぎます（最大1000文字）', 'b19-ai-bot-chat'));
        }
        
        if (empty($message)) {
            wp_send_json_error(__('メッセージを入力してください', 'b19-ai-bot-chat'));
        }
        
        // 危険なコンテンツをチェック
        if (preg_match('/<script|javascript:|on\w+\s*=/i', $message)) {
            wp_send_json_error(__('無効なコンテンツが含まれています', 'b19-ai-bot-chat'));
        }
        
        // レート制限チェック
        $rate_limit = new B19COJP_AI_Bot_Chat_Rate_Limit();
        if (!$rate_limit->check_limit()) {
            wp_send_json_error(__('レート制限に達しました。しばらく時間をおいて再度お試しください。', 'b19-ai-bot-chat'));
        }
        
        // カスタムプロンプトを構築
        $options = get_option('b19cojp_ai_bot_chat_options', array());
        $custom_prompt = $this->build_custom_prompt($options, $message);
        
        // OpenAI API呼び出し
        $openai_client = new B19COJP_AI_Bot_Chat_OpenAI_Client();
        $response = $openai_client->send_message($message, $custom_prompt);
        
        if ($response['success']) {
            $rate_limit->increment();
            wp_send_json_success(array(
                'response' => $response['response'],
                'message' => $response['response']
            ));
        } else {
            wp_send_json_error(array(
                'message' => $response['message']
            ));
        }
    }
    
    public function sanitize_options($input) {
        // 権限チェック
        if (!current_user_can('manage_options')) {
            return get_option('b19cojp_ai_bot_chat_options', array());
        }
        
        // 既存の設定を取得してマージ
        $existing = get_option('b19cojp_ai_bot_chat_options', array());
        $sanitized = $existing; // 既存設定を保持
        
        if (isset($input['enabled'])) {
            $sanitized['enabled'] = (bool) $input['enabled'];
        }
        
        if (isset($input['api_key'])) {
            $sanitized['api_key'] = sanitize_text_field($input['api_key']);
        }
        
        if (isset($input['model'])) {
            $sanitized['model'] = sanitize_text_field($input['model']);
        }
        
        if (isset($input['max_tokens'])) {
            $sanitized['max_tokens'] = absint($input['max_tokens']);
        }
        
        if (isset($input['temperature'])) {
            $sanitized['temperature'] = floatval($input['temperature']);
        }
        
        if (isset($input['rate_limit'])) {
            $sanitized['rate_limit'] = absint($input['rate_limit']);
        }
        
        if (isset($input['bot_name'])) {
            $sanitized['bot_name'] = sanitize_text_field($input['bot_name']);
        }
        
        if (isset($input['initial_message'])) {
            $sanitized['initial_message'] = sanitize_textarea_field($input['initial_message']);
        }
        
        if (isset($input['custom_responses'])) {
            if (!isset($sanitized['custom_responses']) || !is_array($sanitized['custom_responses'])) {
                $sanitized['custom_responses'] = array();
            }
            foreach ($input['custom_responses'] as $key => $value) {
                $sanitized['custom_responses'][sanitize_key($key)] = sanitize_textarea_field($value);
            }
        }
        
        if (isset($input['company_name'])) {
            $sanitized['company_name'] = sanitize_text_field($input['company_name']);
        }
        
        if (isset($input['company_description'])) {
            $sanitized['company_description'] = sanitize_textarea_field($input['company_description']);
        }
        
        if (isset($input['business_hours'])) {
            $sanitized['business_hours'] = sanitize_text_field($input['business_hours']);
        }
        
        if (isset($input['contact_info'])) {
            $sanitized['contact_info'] = sanitize_text_field($input['contact_info']);
        }
        
        if (isset($input['custom_prompt'])) {
            $sanitized['custom_prompt'] = sanitize_textarea_field($input['custom_prompt']);
        }

        // Pro版：スプレッドシート関連オプション
        if (isset($input['spreadsheet_url'])) {
            $sanitized['spreadsheet_url'] = esc_url_raw($input['spreadsheet_url']);
        }
        if (isset($input['sheet_name'])) {
            $sanitized['sheet_name'] = sanitize_text_field($input['sheet_name']);
        }
        if (isset($input['service_account_json'])) {
            $json_test = json_decode($input['service_account_json'], true);
            if (json_last_error() === JSON_ERROR_NONE) {
                $sanitized['service_account_json'] = $input['service_account_json'];
            }


        if (isset($input['sheet_name'])) {
            $sanitized['sheet_name'] = sanitize_text_field($input['sheet_name']);
        }

        if (isset($input['service_account_json'])) {
            // JSONとして有効かチェック
            $json_test = json_decode($input['service_account_json'], true);
            if (json_last_error() === JSON_ERROR_NONE) {
                $sanitized['service_account_json'] = $input['service_account_json'];
            }
        }
            $sanitized['custom_prompt'] = sanitize_textarea_field($input['custom_prompt']);
        }
        
        return $sanitized;
    }
    
    public function activate() {
        $default_options = array(
            'enabled' => true,
            'api_key' => '',
            'model' => 'gpt-4o-mini',
            'max_tokens' => 1000,
            'temperature' => 0.7,
            'rate_limit' => 10,
            'bot_name' => 'AI Bot Chat',
            'initial_message' => 'こんにちは！何かお手伝いできることはありますか？',
            'custom_responses' => array(
                'greeting' => 'こんにちは！お気軽にお声がけください。',
                'help' => 'どのようなことでお手伝いできるでしょうか？',
                'goodbye' => 'ありがとうございました。またお気軽にお声がけください！'
            ),
            'company_name' => '',
            'company_description' => '',
            'business_hours' => '年中無休 24時間対応',
            'contact_info' => '',
            'custom_prompt' => ''
        );
        
        add_option('b19cojp_ai_bot_chat_options', $default_options);
    }
    
    public function deactivate() {
        // プラグイン無効化時の処理
    }
    
    /**
     * カスタムプロンプトを構築
     */
    private function build_custom_prompt($options, $user_message) {
        $prompt = '';
        
        // 会社・お店の情報を追加
        if (!empty($options['company_name'])) {
            $prompt .= "あなたは「" . $options['company_name'] . "」のAIアシスタントです。\n";
        }
        
        if (!empty($options['company_description'])) {
            $prompt .= "会社の情報：\n" . $options['company_description'] . "\n\n";
        }
        
        if (!empty($options['business_hours'])) {
            $prompt .= "営業時間：" . $options['business_hours'] . "\n";
        }
        
        if (!empty($options['contact_info'])) {
            $prompt .= "連絡先：" . $options['contact_info'] . "\n\n";
        }
        
        // カスタムプロンプトが設定されている場合は使用
        if (!empty($options['custom_prompt'])) {
            $prompt .= $options['custom_prompt'] . "\n\n";
        } else {
            // デフォルトのプロンプト
            $prompt .= "あなたは親切で丁寧なAIアシスタントです。\n";
            $prompt .= "ユーザーの質問に対して、正確で分かりやすい回答をしてください。\n\n";
        }
        
        
        // Pro版：スプレッドシートからFAQを追加
        $license_validator = new B19COJP_AI_Bot_Chat_License_Validator();
        if ($license_validator->is_pro_active()) {
            if (!empty($options['spreadsheet_url']) && !empty($options['service_account_json'])) {
                $sheet_name = !empty($options['sheet_name']) ? $options['sheet_name'] : 'FAQ';
                $sheets_client = new B19AIBOC_Google_Sheets_Client(
                    $options['service_account_json'],
                    $options['spreadsheet_url'],
                    $sheet_name
                );
                $faq_prompt = $sheets_client->get_faq_prompt_text();
                if (!empty($faq_prompt)) {
                    $prompt .= $faq_prompt;
                }
            }
        }

        return $prompt;
    }
    
}

// プラグインを初期化
$b19cojp_ai_bot_chat_main = B19COJP_AI_Bot_Chat_Main::get_instance();
