// DOMContentLoadedで即座に初期状態を設定
document.addEventListener('DOMContentLoaded', function() {
    const chatInterface = document.querySelector('.ai-bot-chat-b19-interface');
    if (chatInterface) {
        // CSSクラスに依存して最小化状態を適用
        chatInterface.classList.add('minimized');
        chatInterface.classList.remove('expanded');
        
        // トグルアイコンを設定
        const toggleIcon = chatInterface.querySelector('.ai-bot-chat-b19-toggle-icon');
        if (toggleIcon) {
            toggleIcon.textContent = '+';
        }
        
        console.log('DOMContentLoaded: Chat interface set to minimized state via CSS classes');
    }
});

jQuery(document).ready(function($) {
    'use strict';
    
    // チャットBOTの要素を取得（最初の要素のみを対象）
    const chatInterface = $('.ai-bot-chat-b19-interface').first();
    const chatMessages = chatInterface.find('#ai-bot-chat-b19-messages');
    const chatInput = chatInterface.find('textarea#ai-bot-chat-new-input').first();
    const sendButton = chatInterface.find('button#ai-bot-chat-new-send-button').first();
    const characterCounter = chatInterface.find('#ai-bot-chat-b19-character-counter').first();
    const inputStatus = chatInterface.find('#ai-bot-chat-b19-input-status').first();
    
    // フォールバック用のセレクター（最初の要素のみ）
    const chatInputFallback = $('#ai-bot-chat-new-input').first();
    const sendButtonFallback = $('#ai-bot-chat-new-send-button').first();
    
    console.log('Element counts:', {
        chatInterface: chatInterface.length,
        chatInput: chatInput.length,
        chatInputFallback: chatInputFallback.length,
        sendButton: sendButton.length,
        sendButtonFallback: sendButtonFallback.length
    });
    
    // チャットBOTが存在しない場合は処理を終了
    if (chatInterface.length === 0) {
        console.log('Chat interface not found, skipping initialization');
        return;
    }
    
    console.log('AI Bot Chat Frontend initialized');
    console.log('Chat interface found:', chatInterface.length);
    console.log('Chat input found:', chatInput.length);
    console.log('Send button found:', sendButton.length);
    console.log('Chat input disabled:', chatInput.prop('disabled'));
    console.log('Chat input placeholder:', chatInput.attr('placeholder'));
    
    // 複数のチャットインターフェースが存在する場合は警告
    const allInterfaces = $('.ai-bot-chat-b19-interface');
    if (allInterfaces.length > 1) {
        console.warn('Multiple chat interfaces found:', allInterfaces.length, 'Removing duplicates');
        // 2番目以降の重複を削除
        allInterfaces.slice(1).remove();
        console.log('Duplicate chat interfaces removed');
    }
    
    let isProcessing = false;
    let conversationHistory = [];
    let rateLimitCount = 0;
    let rateLimitResetTime = Date.now() + 60000; // 1分後
    
    // 設定値
    const MAX_MESSAGE_LENGTH = 1000; // 最大文字数
    const MIN_MESSAGE_LENGTH = 1;    // 最小文字数
    
    // 入力フィールドの監視
    chatInput.on('input', function() {
        const message = $(this).val();
        const inputElement = document.getElementById('chat-input');
        console.log('Input event triggered, message:', message);
        console.log('Input event - jQuery val():', message);
        console.log('Input event - direct value:', inputElement ? inputElement.value : 'null');
        console.log('Input event - timestamp:', new Date().toISOString());
        
        const sanitizedMessage = sanitizeInput(message);

        // 入力内容をサニタイズ
        if (message !== sanitizedMessage) {
            $(this).val(sanitizedMessage);
        }
        
        const trimmedMessage = sanitizedMessage.trim();
        const isValid = validateInput(trimmedMessage);
        
        console.log('Input validation:', {
            message: trimmedMessage,
            isValid: isValid,
            isProcessing: isProcessing,
            buttonDisabled: !isValid || isProcessing
        });
        
        // APIキーが設定されていない場合は常に無効化
        if (chatInput.prop('disabled')) {
            sendButton.prop('disabled', true);
        } else {
            sendButton.prop('disabled', !isValid || isProcessing);
        }
        
        // 文字数カウンターとステータス更新
        updateCharacterCounter(trimmedMessage.length);
        updateInputStatus(trimmedMessage);
        
        // テキストエリアの高さを自動調整（CSSのmin-height/max-heightに依存）
        this.style.height = 'auto';
        this.style.height = Math.max(this.scrollHeight, 80) + 'px';
    });
    
    // Enterキーで送信（Shift+Enterで改行）（既存のイベントハンドラーを削除してから追加）
    chatInput.off('keydown').on('keydown', function(e) {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            
            const currentValue1 = chatInput.val();
            const currentValue2 = chatInputFallback.val();
            const currentValue3 = $('textarea#ai-bot-chat-new-input').first().val();
            const currentValue = (currentValue1 || currentValue2 || currentValue3 || '').trim();
            console.log('Enter key pressed, input value:', currentValue);
            console.log('Enter key press - timestamp:', new Date().toISOString());
            console.log('Enter input values:', { currentValue1, currentValue2, currentValue3 });
            console.log('Chat input length on Enter:', chatInput.length);
            
            if (!currentValue || currentValue.trim() === '') {
                console.log('No message to send on Enter');
                return;
            }
            
            if (!sendButton.prop('disabled')) {
                sendMessage();
            }
        }
    });
    
    // 入力サニタイズ関数
    function sanitizeInput(input) {
        if (!input) return '';
        
        // HTMLタグを除去
        let sanitized = input.replace(/<[^>]*>/g, '');
        
        // 危険な文字をエスケープ
        sanitized = sanitized
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#x27;')
            .replace(/\//g, '&#x2F;');
        
        // 文字数制限
        if (sanitized.length > MAX_MESSAGE_LENGTH) {
            sanitized = sanitized.substring(0, MAX_MESSAGE_LENGTH);
        }
        
        return sanitized;
    }
    
    // 文字数カウンター更新
    function updateCharacterCounter(length) {
        const maxLength = parseInt(chatInput.attr('data-max-length')) || 1000;
        const minLength = parseInt(chatInput.attr('data-min-length')) || 1;
        
        characterCounter.text(`${length}/${maxLength} 文字`);
        
        // 文字数に応じて色を変更
        characterCounter.removeClass('warning error');
        if (length > maxLength * 0.9) {
            characterCounter.addClass('error');
        } else if (length > maxLength * 0.8) {
            characterCounter.addClass('warning');
        }
    }
    
    // 入力ステータス更新
    function updateInputStatus(message) {
        const maxLength = parseInt(chatInput.attr('data-max-length')) || 1000;
        const minLength = parseInt(chatInput.attr('data-min-length')) || 1;
        const length = message.length;
        
        inputStatus.removeClass('valid invalid warning');
        
        if (length < minLength) {
            inputStatus.addClass('invalid').text(`最低${minLength}文字必要`);
        } else if (length > maxLength) {
            inputStatus.addClass('invalid').text(`最大${maxLength}文字まで`);
        } else if (hasDangerousContent(message)) {
            inputStatus.addClass('invalid').text('HTMLタグは使用できません');
        } else if (length > maxLength * 0.9) {
            inputStatus.addClass('warning').text('文字数が多くなっています');
        } else {
            inputStatus.addClass('valid').text('送信可能');
        }
    }
    
    // 危険なコンテンツの検出
    function hasDangerousContent(input) {
        const dangerousPatterns = [
            /<script/i,
            /javascript:/i,
            /on\w+\s*=/i,
            /<iframe/i,
            /<object/i,
            /<embed/i,
            /<link/i,
            /<meta/i,
            /<style/i
        ];
        
        return dangerousPatterns.some(pattern => pattern.test(input));
    }
    
    // 入力検証関数
    function validateInput(input) {
        console.log('validateInput called with:', input);
        
        if (!input || input.length < MIN_MESSAGE_LENGTH) {
            console.log('Validation failed: too short or empty');
            return false;
        }
        
        if (input.length > MAX_MESSAGE_LENGTH) {
            console.log('Validation failed: too long');
            return false;
        }
        
        // 危険なパターンをチェック
        const dangerousPatterns = [
            /<script/i,
            /javascript:/i,
            /on\w+\s*=/i,
            /data:/i,
            /vbscript:/i
        ];
        
        for (let pattern of dangerousPatterns) {
            if (pattern.test(input)) {
                return false;
            }
        }
        
        return true;
    }
    
    // 文字数カウンター更新
    function updateCharacterCount(length) {
        let counter = $('.character-counter');
        if (counter.length === 0) {
            counter = $('<div class="character-counter"></div>');
            chatInput.after(counter);
        }
        
        const remaining = MAX_MESSAGE_LENGTH - length;
        const isOverLimit = length > MAX_MESSAGE_LENGTH;
        
        counter.html(`${length}/${MAX_MESSAGE_LENGTH} 文字`);
        
        if (isOverLimit) {
            counter.addClass('over-limit');
        } else {
            counter.removeClass('over-limit');
        }
    }
    
    // 入力エラー表示
    function showInputError() {
        const errorMessage = '入力内容に問題があります。HTMLタグやスクリプトは使用できません。';
        showNotification(errorMessage, 'error');
    }
    
    // 送信ボタンクリック（既存のイベントハンドラーを削除してから追加）
    sendButton.off('click').on('click', function(e) {
        e.preventDefault(); // デフォルトの動作を防ぐ
        
        console.log('Send button clicked');
        console.log('Send button click - timestamp:', new Date().toISOString());
        console.log('Button disabled:', sendButton.prop('disabled'));
        console.log('Input value before:', chatInput.val());
        console.log('Input element:', chatInput[0]);
        console.log('Input disabled:', chatInput.prop('disabled'));
        console.log('Input length:', chatInput.length);
        
        // 入力フィールドが無効化されている場合は警告
        if (chatInput.prop('disabled')) {
            console.log('Input field is disabled - API key not set');
            addMessage('⚠️ APIキーが設定されていません。管理者に設定を依頼してください。', 'bot', true);
            return;
        }
        
        // 送信前に再度値を確認（複数の方法で）
        const currentValue1 = chatInput.val();
        const currentValue2 = chatInputFallback.val();
        const currentValue3 = $('textarea#ai-bot-chat-new-input').first().val();
        const currentValue = (currentValue1 || currentValue2 || currentValue3 || '').trim();
        console.log('Input value at send time:', currentValue);
        console.log('Input values at send time:', { currentValue1, currentValue2, currentValue3 });
        console.log('Chat input length at send time:', chatInput.length);
        
        if (!currentValue || currentValue.trim() === '') {
            console.log('No message to send');
            return;
        }
        
        sendMessage();
    });
    
    // チャットBOT最大化/最小化機能
    const toggleBtn = chatInterface.find('#ai-bot-chat-b19-toggle-btn');
    const toggleIcon = chatInterface.find('.ai-bot-chat-b19-toggle-icon');
    
    // 初期状態を最小化に設定（強制的に）
    chatInterface.removeClass('expanded').addClass('minimized');
    if (toggleIcon.length > 0) {
        toggleIcon.text('+');
    }
    
    // CSSクラスに依存（インラインスタイルは削除）
    
    // 最小化時に不要な要素を非表示
    chatInterface.find('.ai-bot-chat-b19-messages, .ai-bot-chat-b19-input-area, .ai-bot-chat-b19-info').hide();
    
    console.log('Chat interface initialized with minimized state');
    
    // トグルボタンのクリックイベント
    toggleBtn.off('click.chatbot').on('click.chatbot', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        console.log('Toggle button clicked');
        console.log('Current classes:', chatInterface.attr('class'));
        console.log('Is minimized:', chatInterface.hasClass('minimized'));
        console.log('Is expanded:', chatInterface.hasClass('expanded'));
        
        if (chatInterface.hasClass('minimized')) {
            // 最大化
            chatInterface.removeClass('minimized').addClass('expanded');
            // 最大化時に要素を表示
            chatInterface.find('.ai-bot-chat-b19-messages, .ai-bot-chat-b19-input-area, .ai-bot-chat-b19-info').show();
            if (toggleIcon.length > 0) {
                toggleIcon.text('−');
            }
            console.log('Chat interface expanded');
        } else {
            // 最小化
            chatInterface.removeClass('expanded').addClass('minimized');
            // 最小化時に要素を非表示
            chatInterface.find('.ai-bot-chat-b19-messages, .ai-bot-chat-b19-input-area, .ai-bot-chat-b19-info').hide();
            if (toggleIcon.length > 0) {
                toggleIcon.text('+');
            }
            console.log('Chat interface minimized');
        }
    });
    
    // ヘッダークリックでも最大化（最小化時のみ）
    chatInterface.find('.ai-bot-chat-b19-header').off('click.chatbot').on('click.chatbot', function(e) {
        // ボタンクリックは除外
        if ($(e.target).closest('.ai-bot-chat-b19-controls').length) {
            return;
        }
        
        // 最小化時のみ最大化
        if (chatInterface.hasClass('minimized')) {
            chatInterface.removeClass('minimized').addClass('expanded');
            chatInterface.css({
                'height': '500px',
                'width': '350px',
                'border-radius': '10px',
                'background': 'white',
                'overflow': 'visible'
            });
            // 最大化時に要素を表示
            chatInterface.find('.ai-bot-chat-b19-messages, .ai-bot-chat-b19-input-area, .ai-bot-chat-b19-info').show();
            if (toggleIcon.length > 0) {
                toggleIcon.text('−');
            }
            console.log('Chat interface expanded via header click');
        }
    });
    
    // メッセージ送信処理
    function sendMessage() {
        // 複数の方法でメッセージを取得
        const message1 = chatInput.val();
        const message2 = chatInputFallback.val();
        const message3 = $('textarea#ai-bot-chat-b19-input').first().val();
        const message = (message1 || message2 || message3 || '').trim();
        
        console.log('sendMessage called with:', message);
        console.log('Message sources:', { message1, message2, message3 });
        console.log('Chat input length:', chatInput.length);
        console.log('Message length:', message.length);
        console.log('Has dangerous content:', hasDangerousContent(message));
        
        if (!message || isProcessing) {
            console.log('Message empty or processing, returning');
            return;
        }
        
        // 詳細な入力検証
        const maxLength = parseInt(chatInput.attr('data-max-length')) || 1000;
        const minLength = parseInt(chatInput.attr('data-min-length')) || 1;
        
        if (message.length < minLength) {
            console.log('Validation failed: too short');
            addMessage(`メッセージは最低${minLength}文字必要です。`, 'bot', true);
            return;
        }
        
        if (message.length > maxLength) {
            console.log('Validation failed: too long');
            addMessage(`メッセージは最大${maxLength}文字までです。`, 'bot', true);
            return;
        }
        
        if (hasDangerousContent(message)) {
            console.log('Validation failed: dangerous content');
            addMessage('HTMLタグやスクリプトは使用できません。', 'bot', true);
            return;
        }
        
        // 入力検証
        if (!validateInput(message)) {
            showInputError();
            return;
        }
        
        // レート制限チェック
        if (!checkRateLimit()) {
            showRateLimitError();
            return;
        }
        
        // ユーザーメッセージを表示
        addMessage(message, 'user');
        chatInput.val('');
        chatInput.css('height', 'auto');
        sendButton.prop('disabled', true);
        
        // 会話履歴に追加
        conversationHistory.push({
            role: 'user',
            content: message
        });
        
        // AI応答を取得
        getAIResponse(message);
    }
    
    // メッセージ表示 - LINE風吹き出し対応
    function addMessage(content, sender, isError = false) {
        const messageDiv = $('<div>').addClass('ai-bot-chat-b19-message').addClass('ai-bot-chat-b19-' + sender + '-message');
        const messageContent = $('<div>').addClass('ai-bot-chat-b19-message-content');
        const messageTime = $('<div>').addClass('ai-bot-chat-b19-message-time');
        
        if (isError) {
            messageContent.addClass('error-message');
        }
        
        messageContent.html(content);
        messageTime.text(getCurrentTime());
        
        messageDiv.append(messageContent).append(messageTime);
        chatMessages.append(messageDiv);
        
        // スクロールを最下部に
        chatMessages.scrollTop(chatMessages[0].scrollHeight);
    }
    
    // 現在時刻取得
    function getCurrentTime() {
        const now = new Date();
        return now.getHours().toString().padStart(2, '0') + ':' + 
               now.getMinutes().toString().padStart(2, '0');
    }
    
    // AI応答取得
    function getAIResponse(userMessage) {
        isProcessing = true;
        
        // ローディング表示 - LINE風吹き出し対応
        const loadingDiv = $('<div>').addClass('ai-bot-chat-b19-message ai-bot-chat-b19-bot-message loading-message');
        const loadingContent = $('<div>').addClass('ai-bot-chat-b19-message-content');
        loadingContent.html('<div class="loading-dots"><span></span><span></span><span></span></div>');
        loadingDiv.append(loadingContent);
        chatMessages.append(loadingDiv);
        chatMessages.scrollTop(chatMessages[0].scrollHeight);
        
        // APIリクエスト
        console.log('=== AI Response Request ===');
        console.log('Message:', userMessage);
        console.log('b19cojp_ai_bot_chat_ajax object:', typeof b19cojp_ai_bot_chat_ajax, b19cojp_ai_bot_chat_ajax);
        
        if (typeof b19cojp_ai_bot_chat_ajax === 'undefined') {
            console.error('b19cojp_ai_bot_chat_ajax is undefined!');
            $('.loading-message').remove();
            addMessage('エラー: チャットボットの設定が正しく読み込まれていません。ページを再読み込みしてください。', 'bot', true);
            isProcessing = false;
            return;
        }
        
        console.log('AJAX URL:', b19cojp_ai_bot_chat_ajax.ajax_url);
        console.log('Nonce:', b19cojp_ai_bot_chat_ajax.nonce);
        console.log('Conversation History:', conversationHistory);
        
        $.ajax({
            url: b19cojp_ai_bot_chat_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'b19cojp_ai_bot_chat_send_message',
                message: userMessage,
                conversation_history: conversationHistory,
                nonce: b19cojp_ai_bot_chat_ajax.nonce
            },
            success: function(response) {
                console.log('=== AI Response Success ===');
                console.log('Full response:', response);
                console.log('Response success:', response.success);
                console.log('Response data:', response.data);
                
                if (response.success) {
                    // ローディング削除
                    $('.loading-message').remove();
                    
                    // AI応答を表示
                    addMessage(response.data.response, 'bot');
                    
                    // 会話履歴に追加
                    conversationHistory.push({
                        role: 'assistant',
                        content: response.data.response
                    });
                    
                    // レート制限カウント増加
                    rateLimitCount++;
                    
                } else {
                    console.log('=== AI Response Failed ===');
                    console.log('Response failed, message:', response.data.message);
                    $('.loading-message').remove();
                    
                    // APIキー未設定の場合は特別なメッセージを表示
                    if (response.data.message && response.data.message.includes('APIキーが登録されていません')) {
                        addMessage('⚠️ チャットボットの設定が完了していません。\n\n管理者の方へ：\n1. WordPress管理画面にログイン\n2. プラグイン → AI Bot Chat → 設定\n3. OpenAI APIキーを設定してください\n\nAPIキーの取得方法：\nhttps://platform.openai.com/api-keys', 'bot', true);
                    } else {
                        const errorMessage = response.data && response.data.message ? response.data.message : '不明なエラーが発生しました';
                        addMessage('エラーが発生しました: ' + errorMessage, 'bot', true);
                    }
                }
            },
            error: function(xhr, status, error) {
                console.error('=== AI Response Error ===');
                console.error('XHR:', xhr);
                console.error('Status:', status);
                console.error('Error:', error);
                console.error('Response Text:', xhr.responseText);
                $('.loading-message').remove();
                addMessage('通信エラーが発生しました。しばらく時間をおいて再度お試しください。', 'bot', true);
            },
            complete: function() {
                isProcessing = false;
                sendButton.prop('disabled', !chatInput.val().trim());
            }
        });
    }
    
    // レート制限チェック
    function checkRateLimit() {
        const now = Date.now();
        
        // 1分経過していたらリセット
        if (now > rateLimitResetTime) {
            rateLimitCount = 0;
            rateLimitResetTime = now + 60000;
        }
        
        const maxRequests = parseInt(chatInterface.data('rate-limit'));
        return rateLimitCount < maxRequests;
    }
    
    // レート制限エラー表示
    function showRateLimitError() {
        const remainingTime = Math.ceil((rateLimitResetTime - Date.now()) / 1000);
        addMessage(`レート制限に達しました。${remainingTime}秒後に再度お試しください。`, 'bot', true);
    }
    
    // レート制限表示更新
    function updateRateLimitDisplay() {
        const maxRequests = parseInt(chatInterface.data('rate-limit'));
        const remaining = Math.max(0, maxRequests - rateLimitCount);
        
        $('.rate-limit').text(`残りリクエスト: ${remaining}回/分`);
        
        // 残り時間表示
        const remainingTime = Math.ceil((rateLimitResetTime - Date.now()) / 1000);
        if (remainingTime > 0) {
            $('.rate-limit').append(` (${remainingTime}秒後にリセット)`);
        }
    }
    
    // 定期的にレート制限表示を更新
    setInterval(updateRateLimitDisplay, 1000);
    
    // 初期化
    updateRateLimitDisplay();
    
    // エラーメッセージのスタイル
    $('<style>')
        .prop('type', 'text/css')
        .html(`
            .loading-dots {
                display: flex;
                gap: 4px;
                justify-content: center;
                padding: 10px;
            }
            
            .loading-dots span {
                width: 8px;
                height: 8px;
                border-radius: 50%;
                background: #007cba;
                animation: loading 1.4s ease-in-out infinite both;
            }
            
            .loading-dots span:nth-child(1) { animation-delay: -0.32s; }
            .loading-dots span:nth-child(2) { animation-delay: -0.16s; }
            
            @keyframes loading {
                0%, 80%, 100% {
                    transform: scale(0);
                    opacity: 0.5;
                }
                40% {
                    transform: scale(1);
                    opacity: 1;
                }
            }
            
            .error-message {
                background: #f8d7da !important;
                color: #721c24 !important;
                border-color: #f5c6cb !important;
            }
        `)
        .appendTo('head');
});
