<?php
/**
 * Google OAuth認証クライアント
 * ユーザーが選択したスプレッドシートのみアクセス可能
 */

if (!defined('ABSPATH')) {
    exit;
}

class B19AIBOC_Google_OAuth {
    
    private $client_id;
    private $client_secret;
    private $redirect_uri;
    
    // drive.file スコープ = ユーザーが明示的に選択したファイルのみアクセス可能
    private $scopes = array(
        'https://www.googleapis.com/auth/drive.file',
        'https://www.googleapis.com/auth/spreadsheets.readonly'
    );
    
    public function __construct() {
        $options = get_option('ai_bot_chat_options', array());
        $this->client_id = isset($options['google_client_id']) ? $options['google_client_id'] : '';
        $this->client_secret = isset($options['google_client_secret']) ? $options['google_client_secret'] : '';
        $this->redirect_uri = admin_url('admin.php?page=ai-bot-chat&google_oauth_callback=1');
        
        add_action('admin_init', array($this, 'handle_oauth_callback'));
    }
    
    /**
     * Google認証URLを生成
     */
    public function get_auth_url() {
        $params = array(
            'client_id' => $this->client_id,
            'redirect_uri' => $this->redirect_uri,
            'response_type' => 'code',
            'scope' => implode(' ', $this->scopes),
            'access_type' => 'offline',
            'prompt' => 'consent'
        );
        
        return 'https://accounts.google.com/o/oauth2/v2/auth?' . http_build_query($params);
    }
    
    /**
     * OAuthコールバック処理
     */
    public function handle_oauth_callback() {
        if (!isset($_GET['google_oauth_callback']) || !isset($_GET['code'])) {
            return;
        }
        
        if (!current_user_can('manage_options')) {
            return;
        }
        
        $code = sanitize_text_field($_GET['code']);
        $tokens = $this->exchange_code_for_tokens($code);
        
        if ($tokens && isset($tokens['access_token'])) {
            $options = get_option('ai_bot_chat_options', array());
            $options['google_access_token'] = $tokens['access_token'];
            $options['google_refresh_token'] = isset($tokens['refresh_token']) ? $tokens['refresh_token'] : '';
            $options['google_token_expires'] = time() + $tokens['expires_in'];
            
            // ユーザー情報を取得
            $user_info = $this->get_user_info($tokens['access_token']);
            if ($user_info) {
                $options['google_user_email'] = $user_info['email'];
                $options['google_user_name'] = isset($user_info['name']) ? $user_info['name'] : '';
            }
            
            update_option('ai_bot_chat_options', $options);
            
            wp_redirect(admin_url('admin.php?page=ai-bot-chat&google_connected=1'));
            exit;
        } else {
            wp_redirect(admin_url('admin.php?page=ai-bot-chat&google_error=1'));
            exit;
        }
    }
    
    /**
     * 認証コードをトークンに交換
     */
    private function exchange_code_for_tokens($code) {
        $response = wp_remote_post('https://oauth2.googleapis.com/token', array(
            'body' => array(
                'code' => $code,
                'client_id' => $this->client_id,
                'client_secret' => $this->client_secret,
                'redirect_uri' => $this->redirect_uri,
                'grant_type' => 'authorization_code'
            ),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return false;
        }
        
        return json_decode(wp_remote_retrieve_body($response), true);
    }
    
    /**
     * アクセストークンを取得（必要に応じてリフレッシュ）
     */
    public function get_access_token() {
        $options = get_option('ai_bot_chat_options', array());
        
        if (empty($options['google_access_token'])) {
            return false;
        }
        
        // トークンが期限切れの場合はリフレッシュ
        if (isset($options['google_token_expires']) && time() > $options['google_token_expires'] - 60) {
            return $this->refresh_access_token();
        }
        
        return $options['google_access_token'];
    }
    
    /**
     * アクセストークンをリフレッシュ
     */
    private function refresh_access_token() {
        $options = get_option('ai_bot_chat_options', array());
        
        if (empty($options['google_refresh_token'])) {
            return false;
        }
        
        $response = wp_remote_post('https://oauth2.googleapis.com/token', array(
            'body' => array(
                'client_id' => $this->client_id,
                'client_secret' => $this->client_secret,
                'refresh_token' => $options['google_refresh_token'],
                'grant_type' => 'refresh_token'
            ),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $tokens = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($tokens['access_token'])) {
            $options['google_access_token'] = $tokens['access_token'];
            $options['google_token_expires'] = time() + $tokens['expires_in'];
            update_option('ai_bot_chat_options', $options);
            return $tokens['access_token'];
        }
        
        return false;
    }
    
    /**
     * ユーザー情報を取得
     */
    private function get_user_info($access_token) {
        $response = wp_remote_get('https://www.googleapis.com/oauth2/v2/userinfo', array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $access_token
            ),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return false;
        }
        
        return json_decode(wp_remote_retrieve_body($response), true);
    }
    
    /**
     * Google連携を解除
     */
    public function disconnect() {
        $options = get_option('ai_bot_chat_options', array());
        
        // トークンを無効化
        if (!empty($options['google_access_token'])) {
            wp_remote_post('https://oauth2.googleapis.com/revoke?token=' . $options['google_access_token']);
        }
        
        // オプションから削除
        unset($options['google_access_token']);
        unset($options['google_refresh_token']);
        unset($options['google_token_expires']);
        unset($options['google_user_email']);
        unset($options['google_user_name']);
        unset($options['google_spreadsheet_id']);
        unset($options['google_spreadsheet_name']);
        
        update_option('ai_bot_chat_options', $options);
    }
    
    /**
     * 接続済みかチェック
     */
    public function is_connected() {
        $options = get_option('ai_bot_chat_options', array());
        return !empty($options['google_access_token']) && !empty($options['google_refresh_token']);
    }
    
    /**
     * スプレッドシートデータを取得
     */
    public function get_spreadsheet_data($spreadsheet_id, $sheet_name = 'FAQ') {
        $access_token = $this->get_access_token();
        
        if (!$access_token) {
            return array();
        }
        
        // キャッシュをチェック
        $cache_key = 'ai_bot_chat_sheet_' . md5($spreadsheet_id . $sheet_name);
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            return $cached;
        }
        
        $url = "https://sheets.googleapis.com/v4/spreadsheets/{$spreadsheet_id}/values/{$sheet_name}";
        
        $response = wp_remote_get($url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $access_token
            ),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return array();
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!isset($data['values'])) {
            return array();
        }
        
        // 1時間キャッシュ
        set_transient($cache_key, $data['values'], HOUR_IN_SECONDS);
        
        return $data['values'];
    }
}
