<?php
/**
 * Google Sheets API クライアント
 * OAuth認証またはサービスアカウント認証に対応
 */

if (!defined('ABSPATH')) {
    exit;
}

class B19AIBOC_Google_Sheets_Client {
    
    private $spreadsheet_url;
    private $sheet_name;
    private $options;
    
    public function __construct($spreadsheet_url = null, $sheet_name = 'FAQ') {
        $this->options = get_option('b19cojp_ai_bot_chat_options', array());
        $this->spreadsheet_url = $spreadsheet_url ?: (isset($this->options['spreadsheet_url']) ? $this->options['spreadsheet_url'] : '');
        $this->sheet_name = $sheet_name ?: (isset($this->options['sheet_name']) ? $this->options['sheet_name'] : 'FAQ');
    }
    
    /**
     * スプレッドシートからQ&Aデータを取得
     */
    public function get_faq_data() {
        if (empty($this->spreadsheet_url)) {
            return array();
        }
        
        // キャッシュをチェック（1時間キャッシュ）
        $cache_key = 'ai_bot_chat_faq_data_' . md5($this->spreadsheet_url . $this->sheet_name);
        $cached_data = get_transient($cache_key);
        
        if ($cached_data !== false) {
            return $cached_data;
        }
        
        try {
            $spreadsheet_id = $this->extract_spreadsheet_id($this->spreadsheet_url);
            if (!$spreadsheet_id) {
                return array();
            }
            
            // アクセストークンを取得
            $access_token = $this->get_access_token();
            if (!$access_token) {
                return array();
            }
            
            // Google Sheets API v4を使用してデータを取得
            $url = "https://sheets.googleapis.com/v4/spreadsheets/{$spreadsheet_id}/values/" . urlencode($this->sheet_name);
            
            $response = wp_remote_get($url, array(
                'headers' => array(
                    'Authorization' => 'Bearer ' . $access_token,
                ),
                'timeout' => 30,
            ));
            
            if (is_wp_error($response)) {
                return array();
            }
            
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            
            if (!isset($data['values']) || !is_array($data['values'])) {
                return array();
            }
            
            $faq_data = $this->parse_faq_data($data['values']);
            
            // 1時間キャッシュ
            set_transient($cache_key, $faq_data, HOUR_IN_SECONDS);
            
            return $faq_data;
            
        } catch (Exception $e) {
            return array();
        }
    }
    
    /**
     * スプレッドシートIDを抽出
     */
    private function extract_spreadsheet_id($url) {
        $pattern = '/\/spreadsheets\/d\/([a-zA-Z0-9-_]+)/';
        if (preg_match($pattern, $url, $matches)) {
            return $matches[1];
        }
        return false;
    }
    
    /**
     * アクセストークンを取得（OAuth優先、サービスアカウントにフォールバック）
     */
    private function get_access_token() {
        // OAuth認証（優先）
        if (!empty($this->options['google_refresh_token'])) {
            $google_oauth = new B19AIBOC_Google_OAuth();
            $token = $google_oauth->get_access_token();
            if ($token) {
                return $token;
            }
        }
        
        // サービスアカウント認証（フォールバック）
        if (!empty($this->options['service_account_json'])) {
            return $this->get_service_account_token();
        }
        
        return false;
    }
    
    /**
     * サービスアカウントでアクセストークンを取得
     */
    private function get_service_account_token() {
        $service_account = json_decode($this->options['service_account_json'], true);
        if (!$service_account) {
            return false;
        }
        
        $jwt = $this->create_jwt($service_account);
        
        $response = wp_remote_post('https://oauth2.googleapis.com/token', array(
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
            ),
            'body' => array(
                'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
                'assertion' => $jwt,
            ),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        return isset($data['access_token']) ? $data['access_token'] : false;
    }
    
    /**
     * JWTトークンを作成
     */
    private function create_jwt($service_account) {
        $header = json_encode(array(
            'alg' => 'RS256',
            'typ' => 'JWT'
        ));
        
        $now = time();
        $payload = json_encode(array(
            'iss' => $service_account['client_email'],
            'scope' => 'https://www.googleapis.com/auth/spreadsheets.readonly',
            'aud' => 'https://oauth2.googleapis.com/token',
            'iat' => $now,
            'exp' => $now + 3600
        ));
        
        $header_encoded = $this->base64url_encode($header);
        $payload_encoded = $this->base64url_encode($payload);
        
        $signature = '';
        $signature_success = openssl_sign(
            $header_encoded . '.' . $payload_encoded,
            $signature,
            $service_account['private_key'],
            'SHA256'
        );
        
        if (!$signature_success) {
            return false;
        }
        
        $signature_encoded = $this->base64url_encode($signature);
        
        return $header_encoded . '.' . $payload_encoded . '.' . $signature_encoded;
    }
    
    /**
     * Base64URLエンコード
     */
    private function base64url_encode($data) {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
    
    /**
     * FAQデータをパース
     */
    private function parse_faq_data($values) {
        $faq_data = array();
        
        // 最初の行をヘッダーとして扱う
        $headers = isset($values[0]) ? $values[0] : array();
        $question_index = array_search('質問', $headers);
        $answer_index = array_search('回答', $headers);
        
        // ヘッダーが見つからない場合は、デフォルトの列を使用
        if ($question_index === false) $question_index = 0;
        if ($answer_index === false) $answer_index = 1;
        
        // データ行を処理
        for ($i = 1; $i < count($values); $i++) {
            $row = $values[$i];
            
            if (isset($row[$question_index]) && isset($row[$answer_index])) {
                $question = trim($row[$question_index]);
                $answer = trim($row[$answer_index]);
                
                if (!empty($question) && !empty($answer)) {
                    $faq_data[] = array(
                        'question' => $question,
                        'answer' => $answer
                    );
                }
            }
        }
        
        return $faq_data;
    }
    
    /**
     * FAQデータをプロンプト用のテキストに変換
     */
    public function get_faq_prompt_text() {
        $faq_data = $this->get_faq_data();
        
        if (empty($faq_data)) {
            return '';
        }
        
        $prompt_text = "\n\n【よくある質問と回答】\n";
        $prompt_text .= "以下の情報を参考にして、ユーザーの質問に適切に回答してください：\n\n";
        
        foreach ($faq_data as $index => $faq) {
            $prompt_text .= "Q" . ($index + 1) . ": " . $faq['question'] . "\n";
            $prompt_text .= "A" . ($index + 1) . ": " . $faq['answer'] . "\n\n";
        }
        
        $prompt_text .= "上記の情報を参考に、ユーザーの質問に最も適切な回答をしてください。\n";
        $prompt_text .= "該当する質問がある場合は、その回答を基に回答してください。\n";
        $prompt_text .= "該当する質問がない場合は、一般的な知識で回答してください。\n";
        
        return $prompt_text;
    }
}
