<?php
/**
 * ライセンス検証クラス（プラグイン側）
 * 
 * b19.co.jpのAPIと通信してライセンスを検証
 */

if (!defined('ABSPATH')) {
    exit;
}

class B19COJP_AI_Bot_Chat_License_Validator {
    
    private $api_url = 'https://b19.co.jp/wp-json/ai-bot-chat/v1';
    private $option_name = 'b19cojp_ai_bot_chat_license';
    private $cache_duration = 86400; // 24時間
    
    /**
     * ライセンス情報を取得
     */
    public function get_license_info() {
        return get_option($this->option_name, array(
            'license_key' => '',
            'status' => 'inactive',
            'last_check' => 0,
            'expires_at' => '',
            'message' => ''
        ));
    }
    
    /**
     * ライセンスキーを保存
     */
    public function save_license_key($license_key) {
        $license_info = $this->get_license_info();
        $license_info['license_key'] = sanitize_text_field($license_key);
        update_option($this->option_name, $license_info);
    }
    
    /**
     * ライセンスをアクティベート
     * 
     * @param string $license_key ライセンスキー
     * @return array 結果
     */
    public function activate_license($license_key) {
        $domain = $this->get_current_domain();
        
        $response = wp_remote_post($this->api_url . '/activate-license', array(
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'license_key' => $license_key,
                'domain' => $domain
            )),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => '通信エラー: ' . $response->get_error_message()
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (isset($data['success']) && $data['success']) {
            // ライセンス情報を保存
            update_option($this->option_name, array(
                'license_key' => $license_key,
                'status' => 'active',
                'last_check' => time(),
                'expires_at' => isset($data['expires_at']) ? $data['expires_at'] : '',
                'message' => $data['message']
            ));
            
            return array(
                'success' => true,
                'message' => 'ライセンスがアクティベートされました。'
            );
        } else {
            return array(
                'success' => false,
                'message' => isset($data['message']) ? $data['message'] : 'アクティベーションに失敗しました。'
            );
        }
    }
    
    /**
     * ライセンスを検証
     * 
     * @param bool $force_check キャッシュを無視して強制チェック
     * @return bool 有効かどうか
     */
    public function is_license_valid($force_check = false) {
        $license_info = $this->get_license_info();
        
        // ライセンスキーがない場合
        if (empty($license_info['license_key'])) {
            return false;
        }
        
        // キャッシュが有効な場合
        if (!$force_check && $license_info['status'] === 'active') {
            $last_check = isset($license_info['last_check']) ? $license_info['last_check'] : 0;
            if ((time() - $last_check) < $this->cache_duration) {
                return true;
            }
        }
        
        // APIで検証
        return $this->validate_with_api($license_info['license_key']);
    }
    
    /**
     * APIでライセンスを検証
     */
    private function validate_with_api($license_key) {
        $domain = $this->get_current_domain();
        
        $response = wp_remote_post($this->api_url . '/validate-license', array(
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'license_key' => $license_key,
                'domain' => $domain
            )),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            // 通信エラー時はキャッシュを信頼
            $license_info = $this->get_license_info();
            return $license_info['status'] === 'active';
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        $is_valid = isset($data['valid']) && $data['valid'];
        
        // ライセンス情報を更新
        $license_info = $this->get_license_info();
        $license_info['status'] = $is_valid ? 'active' : 'inactive';
        $license_info['last_check'] = time();
        $license_info['message'] = isset($data['message']) ? $data['message'] : '';
        if (isset($data['expires_at'])) {
            $license_info['expires_at'] = $data['expires_at'];
        }
        update_option($this->option_name, $license_info);
        
        return $is_valid;
    }
    
    /**
     * ライセンスを解除
     */
    public function deactivate_license() {
        $license_info = $this->get_license_info();
        
        if (empty($license_info['license_key'])) {
            return array(
                'success' => false,
                'message' => 'ライセンスキーが設定されていません。'
            );
        }
        
        $domain = $this->get_current_domain();
        
        $response = wp_remote_post($this->api_url . '/deactivate-license', array(
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'license_key' => $license_info['license_key'],
                'domain' => $domain
            )),
            'timeout' => 30
        ));
        
        // ローカルのライセンス情報をクリア
        delete_option($this->option_name);
        
        if (is_wp_error($response)) {
            return array(
                'success' => true,
                'message' => 'ローカルのライセンスが解除されました。'
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        return array(
            'success' => true,
            'message' => isset($data['message']) ? $data['message'] : 'ライセンスが解除されました。'
        );
    }
    
    /**
     * 現在のドメインを取得
     */
    private function get_current_domain() {
        $site_url = get_site_url();
        $parsed = parse_url($site_url);
        return isset($parsed['host']) ? $parsed['host'] : '';
    }
    
    /**
     * Pro版が有効かどうか
     */
    public function is_pro_active() {
        return $this->is_license_valid();
    }
}
