<?php
/**
 * OpenAI Client Class
 *
 * External Service: OpenAI API
 * Service Provider: OpenAI, Inc.
 * Terms of Service: https://openai.com/terms/
 * Privacy Policy: https://openai.com/privacy/
 *
 * This class handles all communication with OpenAI's API for AI-powered chat functionality.
 */

if (!defined('ABSPATH')) {
    exit;
}

class B19COJP_AI_Bot_Chat_OpenAI_Client {
    
    private $api_key;
    private $model;
    private $max_tokens;
    private $temperature;
    
    public function __construct() {
        $options = get_option('b19cojp_ai_bot_chat_options', array());
        $this->api_key = isset($options['api_key']) ? $options['api_key'] : '';
        $this->model = isset($options['model']) ? $options['model'] : 'gpt-4o-mini';
        $this->max_tokens = isset($options['max_tokens']) ? $options['max_tokens'] : 1000;
        $this->temperature = isset($options['temperature']) ? $options['temperature'] : 0.7;
    }
    
    public function send_message($message, $custom_prompt = '') {
        if (empty($this->api_key)) {
            return array(
                'success' => false,
                'message' => __('APIキーが登録されていません。管理画面でAPIキーを設定してください。', 'b19-ai-bot-chat')
            );
        }
        
        // メッセージを構築
        $messages = array();
        
        // システムプロンプトを追加
        if (!empty($custom_prompt)) {
            $messages[] = array(
                'role' => 'system',
                'content' => $custom_prompt
            );
        }
        
        // ユーザーメッセージを追加
        $messages[] = array(
            'role' => 'user',
            'content' => $message
        );
        
        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'model' => $this->model,
                'messages' => $messages,
                'max_tokens' => $this->max_tokens,
                'temperature' => $this->temperature,
            )),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => __('通信エラーが発生しました', 'b19-ai-bot-chat')
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (isset($data['choices'][0]['message']['content'])) {
            return array(
                'success' => true,
                'response' => $data['choices'][0]['message']['content']
            );
        } else {
            // エラーメッセージの詳細を確認
            $error_message = __('APIからの応答を取得できませんでした', 'b19-ai-bot-chat');
            
            if (isset($data['error'])) {
                $error_code = $data['error']['code'] ?? '';
                $error_type = $data['error']['type'] ?? '';
                
                if ($error_code === 'insufficient_quota' || $error_type === 'insufficient_quota') {
                    $error_message = __('APIキーの利用枠を確認してください。課金設定や利用制限を確認してください。', 'b19-ai-bot-chat');
                } elseif ($error_code === 'invalid_api_key' || $error_type === 'invalid_api_key') {
                    $error_message = __('APIキーが無効です。正しいAPIキーを設定してください。', 'b19-ai-bot-chat');
                } elseif (isset($data['error']['message'])) {
                    $error_message = $data['error']['message'];
                }
            }
            
            return array(
                'success' => false,
                'message' => $error_message
            );
        }
    }
}
