<?php
/**
 * 管理画面クラス
 */

if (!defined('ABSPATH')) {
    exit;
}

class B19_Post_Gallery_Admin {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
    }
    
    /**
     * 管理メニューを追加
     */
    public function add_admin_menu() {
        add_menu_page(
            'B19 Post Gallery',
            'Post Gallery',
            'manage_options',
            'b19-post-gallery',
            array($this, 'render_settings_page'),
            'dashicons-grid-view',
            30
        );
        
        add_submenu_page(
            'b19-post-gallery',
            '設定',
            '設定',
            'manage_options',
            'b19-post-gallery',
            array($this, 'render_settings_page')
        );
        
        add_submenu_page(
            'b19-post-gallery',
            'レイアウト一覧',
            'レイアウト一覧',
            'manage_options',
            'b19-post-gallery-layouts',
            array($this, 'render_layouts_page')
        );
        
        add_submenu_page(
            'b19-post-gallery',
            '使い方',
            '使い方',
            'manage_options',
            'b19-post-gallery-help',
            array($this, 'render_help_page')
        );
    }
    
    /**
     * 設定を登録
     */
    public function register_settings() {
        register_setting('b19_post_gallery_options', 'b19_post_gallery_options');
    }
    
    /**
     * 設定ページを表示
     */
    public function render_settings_page() {
        $options = get_option('b19_post_gallery_options', array());
        ?>
        <div class="wrap b19-admin-wrap">
            <h1>
                <span class="dashicons dashicons-grid-view"></span>
                B19 Post Gallery 設定
            </h1>
            
            <?php if (!B19_POST_GALLERY_IS_PRO) : ?>
            <div class="b19-admin-pro-banner">
                <div class="b19-pro-banner-content">
                    <span class="b19-pro-badge">Pro</span>
                    <div>
                        <h3>プレミアム版で全機能を解放！</h3>
                        <p>複数カテゴリ選択、左右入れ替えレイアウト、高度なカスタマイズ機能が使えます。</p>
                    </div>
                    <a href="https://b19.co.jp/plugins/post-gallery-pro" class="button button-primary" target="_blank">
                        プレミアム版を見る
                    </a>
                </div>
            </div>
            <?php endif; ?>
            
            <form method="post" action="options.php">
                <?php settings_fields('b19_post_gallery_options'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">デフォルトレイアウト</th>
                        <td>
                            <select name="b19_post_gallery_options[default_layout]">
                                <?php 
                                $layouts = B19_Post_Gallery_Shortcode::get_layouts();
                                foreach ($layouts as $key => $layout) : 
                                    if ($layout['is_pro'] && !B19_POST_GALLERY_IS_PRO) continue;
                                ?>
                                    <option value="<?php echo esc_attr($key); ?>" 
                                        <?php selected(isset($options['default_layout']) ? $options['default_layout'] : 'grid-6-4-right', $key); ?>>
                                        <?php echo esc_html($layout['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <p class="description">ショートコードでレイアウトを指定しない場合のデフォルト</p>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button('設定を保存'); ?>
            </form>
            
            <hr>
            
            <h2>ショートコードの使い方</h2>
            <div class="b19-shortcode-examples">
                <div class="b19-example">
                    <h4>基本的な使い方</h4>
                    <code>[b19_gallery]</code>
                    <p>最新5件の投稿を6:4レイアウトで表示</p>
                </div>
                
                <div class="b19-example">
                    <h4>カテゴリを指定</h4>
                    <code>[b19_gallery category="news"]</code>
                    <p>「news」カテゴリの投稿のみ表示</p>
                </div>
                
                <div class="b19-example">
                    <h4>表示件数を変更</h4>
                    <code>[b19_gallery posts_per_page="10"]</code>
                    <p>10件の投稿を表示</p>
                </div>
                
                <div class="b19-example">
                    <h4>レイアウトを変更</h4>
                    <code>[b19_gallery layout="grid-standard"]</code>
                    <p>標準グリッドレイアウトで表示</p>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * レイアウト一覧ページを表示
     */
    public function render_layouts_page() {
        $layouts = B19_Post_Gallery_Shortcode::get_layouts();
        ?>
        <div class="wrap b19-admin-wrap">
            <h1>
                <span class="dashicons dashicons-layout"></span>
                レイアウト一覧
            </h1>
            
            <div class="b19-layouts-grid">
                <?php foreach ($layouts as $key => $layout) : ?>
                <div class="b19-layout-card <?php echo $layout['is_pro'] ? 'b19-layout-pro' : ''; ?>">
                    <?php if ($layout['is_pro']) : ?>
                        <span class="b19-pro-badge">Pro</span>
                    <?php endif; ?>
                    <div class="b19-layout-preview b19-preview-<?php echo esc_attr($key); ?>">
                        <!-- プレビュー画像またはCSSプレビュー -->
                    </div>
                    <h3><?php echo esc_html($layout['name']); ?></h3>
                    <p><?php echo esc_html($layout['description']); ?></p>
                    <code>[b19_gallery layout="<?php echo esc_attr($key); ?>"]</code>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * ヘルプページを表示
     */
    public function render_help_page() {
        ?>
        <div class="wrap b19-admin-wrap">
            <h1>
                <span class="dashicons dashicons-editor-help"></span>
                使い方ガイド
            </h1>
            
            <div class="b19-help-content">
                <h2>ショートコードパラメータ一覧</h2>
                <table class="widefat">
                    <thead>
                        <tr>
                            <th>パラメータ</th>
                            <th>説明</th>
                            <th>デフォルト値</th>
                            <th>例</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><code>layout</code></td>
                            <td>レイアウトの種類</td>
                            <td>grid-6-4-right</td>
                            <td>grid-standard, grid-list</td>
                        </tr>
                        <tr>
                            <td><code>category</code></td>
                            <td>表示するカテゴリ（スラッグ）</td>
                            <td>なし（全て）</td>
                            <td>news, blog</td>
                        </tr>
                        <tr>
                            <td><code>tag</code></td>
                            <td>表示するタグ（スラッグ）</td>
                            <td>なし（全て）</td>
                            <td>featured, pickup</td>
                        </tr>
                        <tr>
                            <td><code>posts_per_page</code></td>
                            <td>表示する投稿数</td>
                            <td>5</td>
                            <td>3, 10, 20</td>
                        </tr>
                        <tr>
                            <td><code>orderby</code></td>
                            <td>並び順の基準</td>
                            <td>date</td>
                            <td>title, rand, modified</td>
                        </tr>
                        <tr>
                            <td><code>order</code></td>
                            <td>昇順/降順</td>
                            <td>DESC</td>
                            <td>ASC</td>
                        </tr>
                        <tr>
                            <td><code>show_date</code></td>
                            <td>日付を表示</td>
                            <td>true</td>
                            <td>false</td>
                        </tr>
                        <tr>
                            <td><code>show_author</code></td>
                            <td>著者を表示</td>
                            <td>false</td>
                            <td>true</td>
                        </tr>
                        <tr>
                            <td><code>show_excerpt</code></td>
                            <td>抜粋を表示</td>
                            <td>true</td>
                            <td>false</td>
                        </tr>
                        <tr>
                            <td><code>show_video_thumbnail</code></td>
                            <td>動画サムネイルを表示</td>
                            <td>true</td>
                            <td>false</td>
                        </tr>
                    </tbody>
                </table>
                
                <h2>動画サムネイル機能について</h2>
                <p>投稿本文にYouTubeまたはVimeoのURLが含まれている場合、自動的に動画のサムネイル画像を取得して表示します。</p>
                <ul>
                    <li>対応: YouTube, Vimeo</li>
                    <li>再生アイコンが自動で表示されます</li>
                    <li>アイキャッチ画像よりも動画サムネイルが優先されます</li>
                </ul>
            </div>
        </div>
        <?php
    }
}

// 管理画面クラスを初期化
new B19_Post_Gallery_Admin();
