<?php
/**
 * ショートコード処理クラス
 */

if (!defined('ABSPATH')) {
    exit;
}

class B19_Post_Gallery_Shortcode {
    
    /**
     * 利用可能なレイアウト
     */
    public static function get_layouts() {
        $layouts = array(
            // 無料版レイアウト
            'grid-6-4-right' => array(
                'name' => '6:4 左大右小',
                'description' => '左に大きな1記事、右に4つの小記事',
                'is_pro' => false,
            ),
            'grid-standard' => array(
                'name' => '標準グリッド',
                'description' => '均等な3カラムグリッド',
                'is_pro' => false,
            ),
            'grid-list' => array(
                'name' => 'リスト表示',
                'description' => 'シンプルなリスト形式',
                'is_pro' => false,
            ),
            // Pro版レイアウト
            'grid-4-6-left' => array(
                'name' => '4:6 左小右大',
                'description' => '左に4つの小記事、右に大きな1記事',
                'is_pro' => true,
            ),
            'grid-featured' => array(
                'name' => 'フィーチャード',
                'description' => '上に大きな1記事、下に3つの小記事',
                'is_pro' => true,
            ),
            'grid-masonry' => array(
                'name' => 'メイソンリー',
                'description' => 'Pinterest風の可変高さグリッド',
                'is_pro' => true,
            ),
            'grid-timeline' => array(
                'name' => 'タイムライン',
                'description' => '日付ベースのタイムライン表示',
                'is_pro' => true,
            ),
            'grid-card' => array(
                'name' => 'カード',
                'description' => 'シャドウ付きカードデザイン',
                'is_pro' => true,
            ),
        );
        
        return $layouts;
    }
    
    /**
     * ショートコードをレンダリング
     */
    public static function render($atts) {
        $atts = shortcode_atts(array(
            'layout' => 'grid-6-4-right',
            'category' => '',
            'categories' => '',
            'tag' => '',
            'posts_per_page' => 5,
            'orderby' => 'date',
            'order' => 'DESC',
            'show_date' => 'true',
            'show_author' => 'false',
            'show_excerpt' => 'true',
            'excerpt_length' => 55,
            'show_video_thumbnail' => 'true',
        ), $atts, 'b19_gallery');
        
        // レイアウトのPro版チェック
        $layouts = self::get_layouts();
        if (isset($layouts[$atts['layout']]) && $layouts[$atts['layout']]['is_pro'] && !B19_POST_GALLERY_IS_PRO) {
            return self::render_pro_notice($atts['layout'], $layouts[$atts['layout']]['name']);
        }
        
        // 複数カテゴリはPro版のみ
        if (!empty($atts['categories']) && !B19_POST_GALLERY_IS_PRO) {
            return self::render_pro_notice('multi-category', '複数カテゴリ選択');
        }
        
        // クエリ引数を構築
        $query_args = array(
            'post_type' => 'post',
            'posts_per_page' => intval($atts['posts_per_page']),
            'orderby' => sanitize_text_field($atts['orderby']),
            'order' => sanitize_text_field($atts['order']),
            'post_status' => 'publish',
        );
        
        // カテゴリフィルター
        if (!empty($atts['category'])) {
            $query_args['category_name'] = sanitize_text_field($atts['category']);
        }
        
        // タグフィルター
        if (!empty($atts['tag'])) {
            $query_args['tag'] = sanitize_text_field($atts['tag']);
        }
        
        $query = new WP_Query($query_args);
        
        if (!$query->have_posts()) {
            return '<p class="b19-no-posts">表示する投稿がありません。</p>';
        }
        
        ob_start();
        
        echo '<div class="b19-post-gallery b19-layout-' . esc_attr($atts['layout']) . '">';
        
        $post_index = 0;
        while ($query->have_posts()) {
            $query->the_post();
            self::render_post_item($atts, $post_index);
            $post_index++;
        }
        
        echo '</div>';
        
        wp_reset_postdata();
        
        return ob_get_clean();
    }
    
    /**
     * 個別投稿アイテムをレンダリング
     */
    private static function render_post_item($atts, $index) {
        $post_id = get_the_ID();
        $is_featured = ($index === 0 && in_array($atts['layout'], array('grid-6-4-right', 'grid-4-6-left', 'grid-featured')));
        
        // 動画サムネイルをチェック
        $video_thumbnail = '';
        $is_video = false;
        if ($atts['show_video_thumbnail'] === 'true') {
            $video_data = B19_Post_Gallery_Video_Thumbnail::get_video_thumbnail($post_id);
            if ($video_data) {
                $video_thumbnail = $video_data['thumbnail'];
                $is_video = true;
            }
        }
        
        $item_class = 'b19-post-item';
        if ($is_featured) {
            $item_class .= ' b19-featured';
        }
        if ($is_video) {
            $item_class .= ' b19-has-video';
        }
        ?>
        <article class="<?php echo esc_attr($item_class); ?>">
            <a href="<?php the_permalink(); ?>" class="b19-post-link">
                <div class="b19-thumbnail-wrap">
                    <?php if ($is_video && $video_thumbnail) : ?>
                        <img src="<?php echo esc_url($video_thumbnail); ?>" alt="<?php the_title_attribute(); ?>" class="b19-thumbnail">
                        <span class="b19-play-icon">
                            <svg viewBox="0 0 24 24" fill="currentColor"><path d="M8 5v14l11-7z"/></svg>
                        </span>
                    <?php elseif (has_post_thumbnail()) : ?>
                        <?php the_post_thumbnail($is_featured ? 'large' : 'medium', array('class' => 'b19-thumbnail')); ?>
                    <?php else : ?>
                        <div class="b19-no-thumbnail">
                            <span>No Image</span>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="b19-post-content">
                    <h3 class="b19-post-title"><?php the_title(); ?></h3>
                    
                    <?php if ($atts['show_excerpt'] === 'true' && $is_featured) : ?>
                        <p class="b19-post-excerpt">
                            <?php echo wp_trim_words(get_the_excerpt(), intval($atts['excerpt_length']), '...'); ?>
                        </p>
                    <?php endif; ?>
                    
                    <div class="b19-post-meta">
                        <?php if ($atts['show_date'] === 'true') : ?>
                            <span class="b19-post-date"><?php echo get_the_date(); ?></span>
                        <?php endif; ?>
                        
                        <?php if ($atts['show_author'] === 'true') : ?>
                            <span class="b19-post-author"><?php the_author(); ?></span>
                        <?php endif; ?>
                    </div>
                </div>
            </a>
        </article>
        <?php
    }
    
    /**
     * Pro版誘導メッセージ
     */
    private static function render_pro_notice($feature, $feature_name) {
        return sprintf(
            '<div class="b19-pro-notice">
                <div class="b19-pro-notice-inner">
                    <span class="b19-pro-badge">Pro</span>
                    <h4>「%s」はプレミアム版の機能です</h4>
                    <p>より高度なレイアウトと機能をお使いいただくには、プレミアム版へのアップグレードをご検討ください。</p>
                    <a href="https://b19.co.jp/plugins/post-gallery-pro" class="b19-pro-button" target="_blank">
                        プレミアム版を見る
                    </a>
                </div>
            </div>',
            esc_html($feature_name)
        );
    }
    
    /**
     * Gutenbergブロック用レンダリング
     */
    public static function render_block($attributes) {
        $atts = array(
            'layout' => isset($attributes['layout']) ? $attributes['layout'] : 'grid-6-4-right',
            'category' => isset($attributes['category']) ? $attributes['category'] : '',
            'posts_per_page' => isset($attributes['postsPerPage']) ? $attributes['postsPerPage'] : 5,
            'show_date' => isset($attributes['showDate']) ? ($attributes['showDate'] ? 'true' : 'false') : 'true',
            'show_author' => isset($attributes['showAuthor']) ? ($attributes['showAuthor'] ? 'true' : 'false') : 'false',
            'show_excerpt' => isset($attributes['showExcerpt']) ? ($attributes['showExcerpt'] ? 'true' : 'false') : 'true',
            'show_video_thumbnail' => isset($attributes['showVideoThumbnail']) ? ($attributes['showVideoThumbnail'] ? 'true' : 'false') : 'true',
        );
        
        return self::render($atts);
    }
}
