<?php
/**
 * 動画サムネイル処理クラス
 */

if (!defined('ABSPATH')) {
    exit;
}

class B19_Post_Gallery_Video_Thumbnail {
    
    /**
     * 投稿から動画サムネイルを取得
     */
    public static function get_video_thumbnail($post_id) {
        $content = get_post_field('post_content', $post_id);
        
        // YouTube URLを検索
        $youtube_id = self::extract_youtube_id($content);
        if ($youtube_id) {
            return array(
                'type' => 'youtube',
                'id' => $youtube_id,
                'thumbnail' => self::get_youtube_thumbnail($youtube_id),
            );
        }
        
        // Vimeo URLを検索
        $vimeo_id = self::extract_vimeo_id($content);
        if ($vimeo_id) {
            return array(
                'type' => 'vimeo',
                'id' => $vimeo_id,
                'thumbnail' => self::get_vimeo_thumbnail($vimeo_id),
            );
        }
        
        // カスタムフィールドからも検索
        $video_url = get_post_meta($post_id, 'b19_video_url', true);
        if (!empty($video_url)) {
            $youtube_id = self::extract_youtube_id($video_url);
            if ($youtube_id) {
                return array(
                    'type' => 'youtube',
                    'id' => $youtube_id,
                    'thumbnail' => self::get_youtube_thumbnail($youtube_id),
                );
            }
            
            $vimeo_id = self::extract_vimeo_id($video_url);
            if ($vimeo_id) {
                return array(
                    'type' => 'vimeo',
                    'id' => $vimeo_id,
                    'thumbnail' => self::get_vimeo_thumbnail($vimeo_id),
                );
            }
        }
        
        return false;
    }
    
    /**
     * YouTubeのIDを抽出
     */
    private static function extract_youtube_id($content) {
        $patterns = array(
            '/youtube\.com\/watch\?v=([a-zA-Z0-9_-]{11})/',
            '/youtube\.com\/embed\/([a-zA-Z0-9_-]{11})/',
            '/youtu\.be\/([a-zA-Z0-9_-]{11})/',
            '/youtube\.com\/v\/([a-zA-Z0-9_-]{11})/',
        );
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $content, $matches)) {
                return $matches[1];
            }
        }
        
        return false;
    }
    
    /**
     * VimeoのIDを抽出
     */
    private static function extract_vimeo_id($content) {
        $patterns = array(
            '/vimeo\.com\/(\d+)/',
            '/player\.vimeo\.com\/video\/(\d+)/',
        );
        
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $content, $matches)) {
                return $matches[1];
            }
        }
        
        return false;
    }
    
    /**
     * YouTubeサムネイルURLを取得
     */
    private static function get_youtube_thumbnail($video_id) {
        // 高画質サムネイルを優先
        $thumbnail_urls = array(
            'maxresdefault' => "https://img.youtube.com/vi/{$video_id}/maxresdefault.jpg",
            'sddefault' => "https://img.youtube.com/vi/{$video_id}/sddefault.jpg",
            'hqdefault' => "https://img.youtube.com/vi/{$video_id}/hqdefault.jpg",
            'mqdefault' => "https://img.youtube.com/vi/{$video_id}/mqdefault.jpg",
        );
        
        // maxresdefaultが存在するか確認（存在しない場合もある）
        $response = wp_remote_head($thumbnail_urls['maxresdefault']);
        if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
            return $thumbnail_urls['maxresdefault'];
        }
        
        // フォールバック
        return $thumbnail_urls['hqdefault'];
    }
    
    /**
     * Vimeoサムネイルを取得
     */
    private static function get_vimeo_thumbnail($video_id) {
        // キャッシュをチェック
        $cache_key = 'b19_vimeo_thumb_' . $video_id;
        $cached = get_transient($cache_key);
        if ($cached) {
            return $cached;
        }
        
        // Vimeo API
        $response = wp_remote_get("https://vimeo.com/api/v2/video/{$video_id}.json");
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!empty($data[0]['thumbnail_large'])) {
            $thumbnail_url = $data[0]['thumbnail_large'];
            // 24時間キャッシュ
            set_transient($cache_key, $thumbnail_url, DAY_IN_SECONDS);
            return $thumbnail_url;
        }
        
        return false;
    }
}
